/*____________________________________________________________________________
	Copyright (C) 2000 Networks Associates Technology, Inc.
	All rights reserved.
	
	$Id: pgpRPCServer.c,v 1.80.2.6 2001/08/16 23:07:57 pbj Exp $
____________________________________________________________________________*/
#if PGP_WIN32
#include <windows.h>
#endif
#include "pgpPubTypes.h"
#include "pgpErrors.h"
#include "pgpFileSpec.h"
#include "pgpUtilities.h"
#include "pgpKeys.h"
#include "pgpPublicKey.h"
#include "pgpContext.h"
#include "pgpRPCMsg.h"
#include "pgpKeyPriv.h"
#include "pgpRandomPoolPriv.h"
#include "pgpRndSeed.h"

PGPContextRef gCtx = NULL;

#if PGP_DEBUG
void DbgPrintf(char *mesg, ...);
#else
#define DbgPrintf(x)
#endif

void
pgpRPCCreateBEContext()
{
	PGPCustomContextInfo info;
	PGPError err;

	// Make sure we DONT use RPC.
	// Make sure we DONT start the FrontEndThread.
	err = PGPsdkInit(kPGPFlags_ForceLocalExecution|kPGPFlags_SuppressCacheThread);
	if ( IsPGPError( err ) ) {
		return;
	}
	
	pgpClearMemory( &info, sizeof( info ) );

	info.sdkAPIVersion 	= kPGPsdkAPIVersion;
	info.memoryMgr		= PGPGetDefaultMemoryMgr();

	err = PGPNewContextCustom( &info, &gCtx );
	if ( IsPGPError( err ) ) {
		return;
	}
}

/*
 * Messages received by this service.
 * Each of the sRecv_XXX functions must construct the
 *   PGPPackMsg to be returned to the client.
 */
	static void
sRecv_Connect(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPInt32 		client_ver;
	PGPUInt32 		conn_ref, ID, EventHandle;

	client_ver = unpack_int32(pkt);

	/*
	 * Check for clients talking the wrong version. PGPMSG_PROTOCOL_REV
	 * needs to get updated when the protocol changes.
	 */
	if (client_ver != PGPMSG_PROTOCOL_REV) {
		rpkt->status = kPGPError_RPCGarbledMsg;
		return;
	}

	ID = unpack_uint32(pkt);
	EventHandle = unpack_uint32(pkt);
	conn_ref = pgpRPCAllocConnection();

#if PGP_WIN32
	// convert the handle to one we can use from the backend process
	{
		HANDLE		hProcess;

		hProcess = OpenProcess (PROCESS_DUP_HANDLE, FALSE, ID);
		DuplicateHandle (hProcess, (HANDLE)EventHandle, 
				GetCurrentProcess (), (HANDLE*)&EventHandle, 
				0, FALSE, DUPLICATE_SAME_ACCESS);
		CloseHandle (hProcess);
	}

	((pgpRPCconnection *)conn_ref)->EventHandle = EventHandle;

	/*
	 * Fill-in UserName in connection.
	 */
	{
		char			namebuf[256];
		DWORD			namebuflen = sizeof(namebuf);

		if (GetUserName(namebuf, &namebuflen) != 0) {
			PGPByte *cp;
			cp = PGPNewData(PGPGetDefaultMemoryMgr(), namebuflen, 0);
			if (cp == NULL) {
				rpkt->status = kPGPError_UnknownError;
				return;
			}
			pgpCopyMemory(namebuf, cp, namebuflen);
			((pgpRPCconnection *)conn_ref)->UserName = cp;
		}
	}
#else

	((pgpRPCconnection *)conn_ref)->ThreadID = ID;

	/*
	 * Fill-in UserName in connection.
	 * On UNIX/LINUX, use ASCII representation of UID.
	 */
	pgpRPCGetUnixClientAuth((pgpRPCconnection *)conn_ref);

#endif

	pack_int32(rpkt, (PGPUInt32)conn_ref);
	rpkt->status = 0;
}

/* Called when we free context in front end */
	static void
sRecv_Disconnect(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	pgpRPCconnection *conn_ref;

	conn_ref = (pgpRPCconnection *)pkt->connect_ref;
	if( IsntNull( conn_ref->UserName ) )
		PGPFreeData( conn_ref->UserName );
#if PGP_WIN32
	if( conn_ref->EventHandle )
		CloseHandle ( conn_ref->EventHandle );
#endif
	pgpClearMemory( conn_ref, sizeof(*conn_ref) );
	PGPFreeData( conn_ref );
	/* Refresh random seed file on disk when we close a frontend context */
	pgpSaveGlobalRandomPool( );
	rpkt->status = 0;

}

	static void
sRecv_Reconnect(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	pgpRPCconnection *conn_ref;

	conn_ref = (pgpRPCconnection *)pkt->connect_ref;
	if( IsntNull( conn_ref->UserName ) )
		PGPFreeData( conn_ref->UserName );

#if PGP_WIN32
	/*
	 * Fill-in UserName in connection.
	 */
	{
		char		namebuf[256];
		DWORD		namebuflen = sizeof(namebuf);

		if (GetUserName(namebuf, &namebuflen) != 0) {
			PGPByte *cp;
			cp = PGPNewData(PGPGetDefaultMemoryMgr(), namebuflen, 0);
			if (cp == NULL) {
				rpkt->status = kPGPError_UnknownError;
				return;
			}
			pgpCopyMemory(namebuf, cp, namebuflen);
			((pgpRPCconnection *)conn_ref)->UserName = cp;
		}
	}
#else
	/*
	 * Fill-in UserName in connection.
	 * On UNIX/LINUX, use an ASCII representation of the UID.
	 */
	{
		char *name = "jqsmith";
		PGPByte *cp = PGPNewData(PGPGetDefaultMemoryMgr(),
							strlen(name)+1,0);
		pgpCopyMemory(name, cp, strlen(name)+1);
		((pgpRPCconnection *)conn_ref)->UserName = cp;
	}
#endif

	rpkt->status = 0;
}

	static void
sRecv_FetchObjectData(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPInt32 id;
	PGPByte *bufPtr;
	PGPSize bufSize;

	id = unpack_int32(pkt);

	err = pgpFetchObjectData_back(
		gCtx,
		id,
		&bufPtr,
		&bufSize);
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_opaque(rpkt, bufPtr, bufSize);
	if( IsntNull( bufPtr ) )
		PGPFreeData( bufPtr );
}

static void
sRecv_GetKeyByKeyID(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPUInt32 dbid;
	PGPKeyID *keyIDIn;
	PGPBoolean dummyOK, deletedOK;
	PGPError err;
	PGPUInt32 outID;
	PGPSize keyIDSize;

	dbid = unpack_uint32(pkt);
	unpack_opaque(pkt, &keyIDIn, &keyIDSize);
	dummyOK = unpack_bool(pkt);
	deletedOK = unpack_bool(pkt);

	err = pgpGetKeyByKeyID_back(
		gCtx,
		dbid,
		keyIDIn,
		dummyOK,
		deletedOK,
		&outID);
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_uint32(rpkt, outID);
}

static void
sRecv_KeyEncrypt(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 id;
	PGPByte *inbuf;
	PGPSize inbuflen;
	PGPPublicKeyMessageFormat format;
	PGPByte *outbuf;
	PGPSize outbuflen;

	id = unpack_uint32(pkt);
	unpack_opaque(pkt, &inbuf, &inbuflen);
	format = (PGPPublicKeyMessageFormat)unpack_uint32(pkt);

	err = pgpKeyEncrypt_back(
		gCtx,
		id,
		inbuf,
		inbuflen,
		format,
		&outbuf,
		&outbuflen);
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_opaque(rpkt, outbuf, outbuflen);
	if( IsntNull( outbuf ) )
		PGPFreeData( outbuf );
}

static void
sRecv_KeyDecrypt(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 id;
	PGPByte *passphrase;
	PGPSize pplen;
	PGPBoolean hashedPhrase;
	PGPUInt32 cacheTimeOut;
	PGPBoolean cacheGlobal;
	PGPByte *inbuf;
	PGPSize inbuflen;
	PGPPublicKeyMessageFormat format;
	PGPByte *outbuf;
	PGPSize outbuflen;

	id = unpack_uint32(pkt);
	unpack_opaque(pkt, &passphrase, &pplen);
	hashedPhrase = unpack_bool(pkt);
	cacheTimeOut = unpack_uint32(pkt);
	cacheGlobal = unpack_bool(pkt);
	unpack_opaque(pkt, &inbuf, &inbuflen);
	format = (PGPPublicKeyMessageFormat)unpack_uint32(pkt);

	err = pgpKeyDecrypt_back(
		gCtx,
		id,
		passphrase,
		pplen,
		hashedPhrase,
		cacheTimeOut,
		cacheGlobal,
		inbuf,
		inbuflen,
		format,
		&outbuf,
		&outbuflen);
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_opaque(rpkt, outbuf, outbuflen);
	if( IsntNull( outbuf ) )
		PGPFreeData( outbuf );
}

static void
sRecv_KeyVerify(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPByte *inbuf;
	PGPSize inbuflen;
	PGPHashAlgorithm hashalg;
	PGPByte *hash;
	PGPSize hashlen;
	PGPPublicKeyMessageFormat format;
	PGPInt32 n;
	PGPUInt32 id;

	id = unpack_uint32(pkt);
	unpack_opaque(pkt, &inbuf, &inbuflen);
	hashalg = (PGPHashAlgorithm) unpack_uint32(pkt);
	unpack_opaque(pkt, &hash, &hashlen);
	format = (PGPPublicKeyMessageFormat)unpack_uint32(pkt);

	n = pgpKeyVerify_back(
		gCtx,
		id,
		inbuf,
		inbuflen,
		hashalg,
		hash,
		hashlen,
		format);
	pack_int32(rpkt, n);
	rpkt->status = kPGPError_NoErr;
}

static void
sRecv_KeySign(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 id;
	PGPByte *passphrase;
	PGPSize pplen;
	PGPBoolean hashedPhrase;
	PGPUInt32 cacheTimeOut;
	PGPBoolean cacheGlobal;
	PGPHashAlgorithm hashalg;
	PGPByte *hash;
	PGPSize hashlen;
	PGPPublicKeyMessageFormat format;
	PGPByte *outbuf;
	PGPSize outbuflen;

	id = unpack_uint32(pkt);
	unpack_opaque(pkt, &passphrase, &pplen);
	hashedPhrase = unpack_bool(pkt);
	cacheTimeOut = unpack_uint32(pkt);
	cacheGlobal = unpack_bool(pkt);
	hashalg = (PGPHashAlgorithm)unpack_int32(pkt);
	unpack_opaque(pkt, &hash, &hashlen);
	format = (PGPPublicKeyMessageFormat)unpack_uint32(pkt);

	err = pgpKeySign_back(
		gCtx,
		id,
		passphrase,
		pplen,
		hashedPhrase,
		cacheTimeOut,
		cacheGlobal,
		hashalg,
		hash,
		hashlen,
		format,
		&outbuf,
		&outbuflen);
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_opaque(rpkt, (PGPByte *)outbuf, outbuflen);
	if( IsntNull( outbuf ) )
		PGPFreeData( outbuf );
}

static void
sRecv_SecPassphraseOK(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPUInt32 id;
	PGPByte *passphrase;
	PGPSize pplen;
	PGPBoolean hashedPhrase, b;
	PGPUInt32 cacheTimeOut;
	PGPBoolean cacheGlobal;

	id = unpack_uint32(pkt);
	unpack_opaque(pkt, &passphrase, &pplen);
	hashedPhrase = unpack_bool(pkt);
	cacheTimeOut = unpack_uint32(pkt);
	cacheGlobal = unpack_bool(pkt);

	b = pgpSecPassphraseOK_back(
		gCtx,
		id,
		passphrase,
		pplen,
		hashedPhrase,
		cacheTimeOut,
		cacheGlobal);

	pack_bool(rpkt, b);

	rpkt->status = kPGPError_NoErr;
}

static void
sRecv_PurgePassphraseCache(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;

	pkt;
	err = pgpPurgePassphraseCache_back(gCtx);

	rpkt->status = err;
}

static void
sRecv_KeyMaxSizes(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 id;
	PGPUInt32 maxEncryption=0, maxDecryption=0, maxSignature=0;
	PGPBoolean needMaxEncryption, needMaxDecryption, needMaxSignature;
	PGPPublicKeyMessageFormat format;

	id = unpack_uint32(pkt);
	needMaxEncryption = unpack_bool(pkt);
	needMaxDecryption = unpack_bool(pkt);
	needMaxSignature = unpack_bool(pkt);
	format = (PGPPublicKeyMessageFormat)unpack_uint32(pkt);

	err = pgpKeyMaxSizes_back(
		gCtx,
		id,
		(needMaxEncryption ? &maxEncryption : NULL),
		(needMaxDecryption ? &maxDecryption : NULL),
		(needMaxSignature  ? &maxSignature  : NULL),
		format);
	rpkt->status = err;
	if (IsPGPError(err)) return;

	pack_uint32(rpkt, maxEncryption);
	pack_uint32(rpkt, maxDecryption);
	pack_uint32(rpkt, maxSignature);
}

static void
sRecv_FetchKeyInfo(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPInt32 id;
	PGPByte *bufPtr;
	PGPSize bufSize;

	id = unpack_int32(pkt);

	err = pgpFetchKeyInfo_back(
		gCtx,
		id,
		&bufPtr,
		&bufSize);
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_opaque(rpkt, bufPtr, bufSize);
	if( IsntNull( bufPtr ) )
		PGPFreeData( bufPtr );
}

static void
sRecv_OpenKeyDBFile(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPInt32 flags;
	PGPUInt32 kdbid, numKeys, *keyBuffer;
	PGPError err;
	PFLFileSpecRef pub, priv;
	PGPSize		bufSize;
	
	flags = unpack_int32(pkt);
	pub = unpack_filespec(pkt, PGPPeekContextMemoryMgr(gCtx) );
	priv = unpack_filespec(pkt, PGPPeekContextMemoryMgr(gCtx) );

	err = pgpOpenKeyDBFile_back(
		gCtx,
		(PGPOpenKeyDBFileOptions) flags,
		pub,
		priv,
		&kdbid,
		&numKeys,
		&keyBuffer,
		&bufSize );
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_int32(rpkt, kdbid);
	pack_int32(rpkt, numKeys);
	pack_opaque(rpkt, (PGPByte *)keyBuffer, bufSize);
	if( IsntNull( keyBuffer ) )
		PGPFreeData( keyBuffer );
}

static void
sRecv_KeyDBFlush(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 id;
	PGPUInt32 *changedkeylist;
	PGPSize changedkeylistsize;

	id = unpack_uint32(pkt);

	err = pgpKeyDBFlush_back(gCtx,
		id, &changedkeylist,
		&changedkeylistsize );
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_opaque(rpkt, changedkeylist, changedkeylistsize);
	if( IsntNull( changedkeylist ) )
		PGPFreeData( changedkeylist );
}

static void
sRecv_FreeKeyDB(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPUInt32 id = unpack_uint32(pkt);

	pgpFreeKeyDB_back(gCtx, id);

	rpkt->status = kPGPError_NoErr;
}

static void
sRecv_SetKeyEnabled(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 id;
	PGPBoolean enable;

	id = unpack_uint32(pkt);
	enable = unpack_bool(pkt);

	err = pgpSetKeyEnabled_back(gCtx, id, enable);
	rpkt->status = err;
	if (IsPGPError(err)) return;
}

static void
sRecv_SetKeyAxiomatic(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 id;
	PGPBoolean setAxiomatic;
	PGPBoolean checkPassphrase;
	PGPByte *passphrase;
	PGPSize passphraseLength;
	PGPBoolean hashedPhrase;
	PGPUInt32 cacheTimeOut;
	PGPBoolean cacheGlobal;

	id = unpack_uint32(pkt);
	setAxiomatic = unpack_bool(pkt);
	checkPassphrase = unpack_bool(pkt);
	unpack_opaque(pkt, &passphrase, &passphraseLength);
	hashedPhrase = unpack_bool(pkt);
	cacheTimeOut = unpack_uint32(pkt);
	cacheGlobal = unpack_bool(pkt);

	err = pgpSetKeyAxiomatic_back(gCtx, id, setAxiomatic, checkPassphrase,
		(char *) passphrase, passphraseLength, hashedPhrase, cacheTimeOut, cacheGlobal);
	rpkt->status = err;
	if (IsPGPError(err)) return;
}

static void
sRecv_PropagateTrust(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 setid, *keylist;
	PGPSize keylistsize;
	PGPUInt32 altid;
	PGPUInt32 timenow;
	PGPUInt32 *changedkeylist;
	PGPSize changedkeylistsize;

	setid = unpack_uint32(pkt);
	unpack_opaque_alloc(pkt, &keylist, &keylistsize);
	altid = unpack_uint32(pkt);
	timenow = unpack_uint32(pkt);

	err = pgpPropagateTrust_back(gCtx,
		setid, keylist, keylistsize, altid,
		timenow, &changedkeylist, &changedkeylistsize );
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_opaque(rpkt, (PGPByte *)changedkeylist, changedkeylistsize);
	if( IsntNull( changedkeylist ) )
		PGPFreeData( changedkeylist );
}

static void
sRecv_CheckKeyRingSigs(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 setid, *keylist;
	PGPSize keylistsize;
	PGPUInt32 altid;
	PGPBoolean checkAll;
	PGPUInt32 *changedkeylist;
	PGPSize changedkeylistsize;

	setid = unpack_uint32(pkt);
	unpack_opaque_alloc(pkt, &keylist, &keylistsize);
	altid = unpack_uint32(pkt);
	checkAll = unpack_bool(pkt);

	err = pgpCheckKeyRingSigs_back(gCtx,
			setid, keylist, keylistsize, altid, checkAll,
			&changedkeylist, &changedkeylistsize);
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_opaque(rpkt, (PGPByte *)changedkeylist, changedkeylistsize);
	if( IsntNull( changedkeylist ) )
		PGPFreeData( changedkeylist );
}

static void
sRecv_PrepareToCheckKeyRingSigs(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 setid, *keylist;
	PGPSize keylistsize;
	PGPUInt32 altid;
	PGPBoolean checkAll;
	PGPUInt32 *changedkeylist;
	PGPSize changedkeylistsize;
	PGPUInt32 nsigs;

	setid = unpack_uint32(pkt);
	unpack_opaque_alloc(pkt, &keylist, &keylistsize);
	altid = unpack_uint32(pkt);
	checkAll = unpack_bool(pkt);

	err = pgpPrepareToCheckKeyRingSigs_back(gCtx,
		setid, keylist, keylistsize, altid, checkAll,
		&nsigs, &changedkeylist, &changedkeylistsize);
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_uint32(rpkt, nsigs);
	pack_opaque(rpkt, (PGPByte *)changedkeylist, changedkeylistsize);
	if( IsntNull( changedkeylist ) )
		PGPFreeData( changedkeylist );
}

static void
sRecv_CheckSig(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 setid, altid, objid;
	PGPBoolean checkAll, revocationonly;
	PGPBoolean handled, changed, verified;

	setid = unpack_uint32(pkt);
	objid = unpack_uint32(pkt);
	altid = unpack_uint32(pkt);
	checkAll = unpack_bool(pkt);
	revocationonly = unpack_bool(pkt);

	err = pgpCheckSig_back(gCtx,
				setid, objid, altid, checkAll, revocationonly,
				&handled, &changed, &verified);
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_bool(rpkt, handled);
	pack_bool(rpkt, changed);
	pack_bool(rpkt, verified);
}

static void
sRecv_DoGenerateKey(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 keydbID;
	PGPUInt32 masterkeyID;
	PGPByte pkalg;
	PGPUInt32 bits;
	PGPTime creationDate;
	PGPUInt16 expirationDays;
	PGPByte *name, *passphrase, *masterpass;
	PGPSize namelen, passphraseLength, masterpassLength;
	PGPUInt32 cacheTimeOut;
	PGPBoolean cacheGlobal;
	PGPBoolean passphraseIsKey;
	PGPUInt32 *adkkeylist;
	PGPSize adkkeylistsize;
	PGPByte adkclass;
	PGPEventHandlerProcPtr progress;
	PGPUserValue userValue;
	PGPBoolean fastgen, checkentropy;
	PGPBoolean usetoken;
	PGPUInt32 tokenid;
	PGPUInt32 *rakkeylist;
	PGPSize rakkeylistsize;
	PGPByte rakclass;
	PGPCipherAlgorithm *prefalg;
	PGPSize prefalgLength;
	PGPByte * prefkeyserv;
	PGPSize prefkeyservLength;
	PGPUInt32 keyflags;
	PGPBoolean fkeyflags;
	PGPUInt32 keyservprefs;
	PGPBoolean fkeyservprefs;
	PGPUInt32 *newobjs;
	PGPSize newobjsLength;
	PGPUInt32 newKey;
	
	keydbID = unpack_uint32(pkt);
	masterkeyID = unpack_uint32(pkt);
	pkalg = unpack_int32(pkt);
	bits = unpack_uint32(pkt);
	creationDate = unpack_uint32(pkt);
	expirationDays = (PGPUInt16)unpack_uint32(pkt);
	unpack_opaque(pkt, &name, &namelen);
	unpack_opaque(pkt, &passphrase, &passphraseLength);
	passphraseIsKey = unpack_bool(pkt);
	unpack_opaque(pkt, &masterpass, &masterpassLength);
	cacheTimeOut = unpack_uint32(pkt);
	cacheGlobal = unpack_bool(pkt);
	progress = (PGPEventHandlerProcPtr)unpack_uint32(pkt);
	userValue = (PGPUserValue)unpack_uint32(pkt);
	fastgen = unpack_bool(pkt);
	checkentropy = unpack_bool(pkt);
	usetoken = unpack_bool(pkt);
	tokenid = unpack_uint32(pkt);
	unpack_opaque_alloc(pkt, &adkkeylist, &adkkeylistsize);
	adkclass = unpack_byte(pkt);
	unpack_opaque_alloc(pkt, &rakkeylist, &rakkeylistsize);
	rakclass = unpack_byte(pkt);
	unpack_opaque(pkt, &prefalg, &prefalgLength);
	unpack_opaque(pkt, &prefkeyserv, &prefkeyservLength);
	keyflags = unpack_uint32(pkt);
	fkeyflags = unpack_bool(pkt);
	keyservprefs = unpack_uint32(pkt);
	fkeyservprefs = unpack_bool(pkt);

	err = pgpDoGenerateKey_back(gCtx,
			keydbID, masterkeyID,
			pkalg, bits, creationDate, expirationDays, (char *) name, namelen,
			(char *) passphrase, passphraseLength,
			passphraseIsKey,
			(char *) masterpass, masterpassLength,
			cacheTimeOut, cacheGlobal, NULL /* progress */, userValue,
			fastgen, checkentropy, usetoken, tokenid,
			adkkeylist, adkkeylistsize,
			adkclass, rakkeylist, rakkeylistsize, rakclass,
			prefalg, prefalgLength, prefkeyserv, prefkeyservLength, keyflags, fkeyflags,
			keyservprefs, fkeyservprefs, &newobjs, &newobjsLength, &newKey);

	rpkt->status = err;
	if (IsPGPError(err)) return;

	pack_opaque(rpkt, newobjs, newobjsLength);
	if( IsntNull( newobjs ) )
		PGPFreeData( newobjs );
	pack_int32(rpkt, newKey);
}

static void
sRecv_CreateKeypair(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 keydbID;
	PGPByte *name, *passphrase;
	PGPSize namelen, passphraseLength;
	PGPByte *seckeyBuf, *keyspecBuf;
	PGPSize seckeySize, keyspecSize;
	PGPUInt32 cacheTimeOut;
	PGPBoolean cacheGlobal;
	PGPBoolean passphraseIsKey;
	PGPUInt32 *adkkeylist;
	PGPSize adkkeylistsize;
	PGPByte adkclass;
	PGPUInt32 *rakkeylist;
	PGPSize rakkeylistsize;
	PGPByte rakclass;
	PGPUInt32 *newobjs;
	PGPSize newobjsLength;
	PGPUInt32 newkeyid;
	
	keydbID = unpack_uint32(pkt);
	unpack_opaque_alloc(pkt, &seckeyBuf, &seckeySize);
	unpack_opaque_alloc(pkt, &keyspecBuf, &keyspecSize);
	unpack_opaque(pkt, &name, &namelen);
	unpack_opaque_alloc(pkt, &adkkeylist, &adkkeylistsize);
	adkclass = unpack_byte(pkt);
	unpack_opaque_alloc(pkt, &rakkeylist, &rakkeylistsize);
	rakclass = unpack_byte(pkt);
	unpack_opaque(pkt, &passphrase, &passphraseLength);
	passphraseIsKey = unpack_bool(pkt);
	cacheTimeOut = unpack_uint32(pkt);
	cacheGlobal = unpack_bool(pkt);

	err = pgpCreateKeypair_back( gCtx, keydbID,
					seckeyBuf, seckeySize, keyspecBuf, keyspecSize,
					(char *)name, namelen,
					adkkeylist, adkkeylistsize, adkclass,
					rakkeylist, rakkeylistsize, rakclass,
					(char *)passphrase, passphraseLength,
					passphraseIsKey, cacheTimeOut, cacheGlobal,
					&newobjs, &newobjsLength, &newkeyid);

	rpkt->status = err;
	if (IsPGPError(err)) return;

	pack_opaque(rpkt, newobjs, newobjsLength);
	if( IsntNull( newobjs ) )
		PGPFreeData( newobjs );
	pack_int32(rpkt, newkeyid);
}

static void
sRecv_CreateSubkeypair(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 masterkeyID;
	PGPByte *passphrase;
	PGPSize passphraseLength;
	PGPByte *seckeyBuf, *keyspecBuf;
	PGPSize seckeySize, keyspecSize;
	PGPUInt32 cacheTimeOut;
	PGPBoolean cacheGlobal;
	PGPBoolean passphraseIsKey;
	PGPUInt32 *newobjs;
	PGPSize newobjsLength;
	PGPUInt32 newkeyid;
	
	masterkeyID = unpack_uint32(pkt);
	unpack_opaque_alloc(pkt, &seckeyBuf, &seckeySize);
	unpack_opaque_alloc(pkt, &keyspecBuf, &keyspecSize);
	unpack_opaque(pkt, &passphrase, &passphraseLength);
	passphraseIsKey = unpack_bool(pkt);
	cacheTimeOut = unpack_uint32(pkt);
	cacheGlobal = unpack_bool(pkt);

	err = pgpCreateSubkeypair_back( gCtx, masterkeyID,
							seckeyBuf, seckeySize,
							keyspecBuf, keyspecSize,
							(char *)passphrase, passphraseLength,
							passphraseIsKey, cacheTimeOut, cacheGlobal,
							&newobjs, &newobjsLength, &newkeyid);

	rpkt->status = err;
	if (IsPGPError(err)) return;

	pack_opaque(rpkt, newobjs, newobjsLength);
	if( IsntNull( newobjs ) )
		PGPFreeData( newobjs );
	pack_int32(rpkt, newkeyid);
}

static void
sRecv_CountTokens(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 ntokens;

	err = pgpCountTokens_back(gCtx, &ntokens );
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_uint32(rpkt, ntokens);
}

static void
sRecv_GetTokenInfo(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 tokNumber;
    PGPUInt32 client_size;
    PGPTokenInfo i;

	tokNumber = unpack_uint32(pkt);
    i.size = client_size = unpack_uint32(pkt);

	err = pgpGetTokenInfo_back(gCtx, tokNumber, &i );

    if( ! IsPGPError(err) )  {
        /* Serialize */
        pack_uint32(rpkt, i.size);

        /* Begin of base structure PGPTokenInfo */
        pack_bool(rpkt, i.bRsa);

        pack_uint32(rpkt, i.minRsaKeySize);
        pack_uint32(rpkt, i.maxRsaKeySize);
        pack_uint32(rpkt, i.minPinLen);
        pack_uint32(rpkt, i.maxPinLen);

        pack_opaque(rpkt, i.manufacturerID, strlen(i.manufacturerID)+1);
	    pack_opaque(rpkt, i.model, strlen(i.model)+1);
	    pack_opaque(rpkt, i.serialNumber, strlen(i.serialNumber)+1);

		pack_uint32(rpkt, i.numPubKeys);
		pack_uint32(rpkt, i.numPrivKeys);

        /* End of base structure PGPTokenInfo, therefore: */
        pgpAssert( i.size == 0 || client_size == i.size/*SDK size*/ );
    }

	rpkt->status = err;
}

static void
sRecv_DeleteKeyOnToken(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPUInt32	keydbID;
    PGPKeyID    *keyID;
	PGPSize		l;
	PGPUInt32	tokenNumber;
	PGPByte		*pin;
	PGPSize		pinLen;

	keydbID = unpack_uint32(pkt);
	unpack_opaque(pkt, &keyID, &l);
	tokenNumber = unpack_uint32(pkt);
	unpack_opaque(pkt, &pin, &pinLen);

	if( l >= sizeof(PGPKeyID) )
		rpkt->status = pgpDeleteKeyOnToken_back( gCtx, keydbID, keyID, tokenNumber, pin, pinLen );
	else
		rpkt->status = kPGPError_RPCGarbledMsg;
}


static void
sRecv_WipeToken(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 tokNumber;
	PGPByte *passphrase;
	PGPSize passphraseLength;

	tokNumber = unpack_uint32(pkt);
	unpack_opaque(pkt, &passphrase, &passphraseLength);
	err = pgpWipeToken_back(gCtx, tokNumber, passphrase, passphraseLength );
	rpkt->status = err;
}

static void
sRecv_TokenPassphraseIsValid(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 tokNumber;
	PGPByte *passphrase;
	PGPSize passphraseLength;

	tokNumber = unpack_uint32(pkt);
	unpack_opaque(pkt, &passphrase, &passphraseLength);
	err = pgpTokenPassphraseIsValid_back(gCtx, tokNumber,
										 passphrase, passphraseLength );
	rpkt->status = err;
}

static void
sRecv_CopyKeyToToken(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 id;
	PGPUInt32 tokNumber;
	PGPByte *passphrase;
	PGPSize passphraseLength;
	PGPBoolean hashedPhrase;
	PGPByte *PIN;
	PGPSize PINlength;
	PGPUInt32 cacheTimeOut;
	PGPBoolean cacheGlobal;
	PGPBoolean isMaster;

	id = unpack_int32(pkt);
	tokNumber = unpack_uint32(pkt);
	isMaster = unpack_bool(pkt);
	unpack_opaque(pkt, &passphrase, &passphraseLength);
	hashedPhrase = unpack_bool(pkt);
	unpack_opaque(pkt, &PIN, &PINlength);
	cacheTimeOut = unpack_uint32(pkt);
	cacheGlobal = unpack_bool(pkt);

	err = pgpCopyKeyToToken_back(gCtx,
				id, tokNumber, isMaster,
				(char *) passphrase, passphraseLength, hashedPhrase,
				(char *)PIN, PINlength, cacheTimeOut, cacheGlobal);
	rpkt->status = err;
}

static void
sRecv_TokenImportX509(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
    PGPKeyID    keyID = { 0 };
	PGPByte     *x509;
	PGPSize     x509_len;
    PGPByte     *password;
    PGPSize     passwordLength;
    PGPByte     *userID;
    PGPSize     userID_len;

    *(PGPUInt32*)&keyID = unpack_uint32(pkt);
    ((PGPUInt32*)&keyID)[1] = unpack_uint32(pkt);

    unpack_opaque(pkt, &userID, &userID_len);
	unpack_opaque(pkt, &password, &passwordLength);
	unpack_opaque(pkt, &x509, &x509_len);

	rpkt->status = pgpTokenImportX509_back(gCtx, (PGPByte*)&keyID, 
        userID,     userID_len, 
        x509,       x509_len, 
        password,   passwordLength );
}


static void
sRecv_TokenPutKeyContainer(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
    PGPKeyID    keyID = { 0 };
    PGPByte     *cont;
    PGPSize     contSize;
    const PGPByte     *password;
    PGPSize     passwordSize;

    *(PGPUInt32*)&keyID = unpack_uint32(pkt);
    ((PGPUInt32*)&keyID)[1] = unpack_uint32(pkt);

    unpack_opaque(pkt, &cont, &contSize);
    unpack_opaque(pkt, (PGPByte**)&password, &passwordSize);

    /* call to pgpTokenPutKeyContainer_internal */
	rpkt->status = pgpTokenPutKeyContainer_back( 
        gCtx, (const PGPByte*)&keyID, 
        password, passwordSize, 
        cont, contSize );
}

static void
sRecv_TokenGetKeyContainer(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
    PGPKeyID		keyID = { 0 };
    PGPByte			*cont;
    PGPSize			contSize;
    const PGPByte   *password;
    PGPSize			passwordSize;
    PGPByte			*contOut;
    PGPSize			contOutSize;

    *(PGPUInt32*)&keyID = unpack_uint32(pkt);
    ((PGPUInt32*)&keyID)[1] = unpack_uint32(pkt);

    unpack_opaque(pkt, &cont, &contSize);
    unpack_opaque(pkt, (PGPByte**)&password, &passwordSize);

    /* call to pgpTokenGetKeyContainer_internal */
	rpkt->status = pgpTokenGetKeyContainer_back( 
        gCtx, (const PGPByte*)&keyID, 
        password, passwordSize, 
        &contOut, &contOutSize );

    /* Reply */
    if( ! IsPGPError(rpkt->status) )  {
        pack_opaque(rpkt, contOut, contOutSize);
    }

    if( !IsNull( contOut ) )
        PGPFreeData( contOut );     /* Allocated in the gCtx */
}

static void
sRecv_SetPKCS11DrvFile(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPByte *module;
	PGPSize moduleLength=0;

	unpack_opaque(pkt, &module, &moduleLength);

    if( moduleLength == 0 )  
		module = NULL;

    err = pgpSetPKCS11DrvFile_back( module );

	rpkt->status = err;
}


static void
sRecv_AddUserID(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 id;
	PGPBoolean isAttribute;
	PGPAttributeType attributeType;
	PGPByte *userIDData, *passphrase;
	PGPSize userIDLength, passphraseLength;
	PGPBoolean hashedPhrase;
	PGPUInt32 cacheTimeOut;
	PGPBoolean cacheGlobal;
	PGPUInt32 *newobjs;
	PGPSize newobjsLength;

	id = unpack_int32(pkt);
	isAttribute = unpack_bool(pkt);
	attributeType = (PGPAttributeType) unpack_int32(pkt);
	unpack_opaque(pkt, &userIDData, &userIDLength);
	unpack_opaque(pkt, &passphrase, &passphraseLength);
	hashedPhrase = unpack_bool(pkt);
	cacheTimeOut = unpack_uint32(pkt);
	cacheGlobal = unpack_bool(pkt);

	err = pgpAddUserID_back(gCtx,
				id, isAttribute, attributeType,
				(char *) userIDData, userIDLength,
				(char *) passphrase, passphraseLength,
				hashedPhrase, cacheTimeOut, cacheGlobal, 
				&newobjs, &newobjsLength);
	rpkt->status = err;
	if (IsPGPError(err)) return;

	pack_opaque(rpkt, (PGPByte *)newobjs, newobjsLength);
	if( IsntNull( newobjs ) )
		PGPFreeData( newobjs );
}

static void
sRecv_CertifyUserID(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 userid, certifying_keyid;
	PGPByte *passphrase;
	PGPSize passphraseLength;
	PGPBoolean hashedPhrase;
	PGPUInt32 cacheTimeOut;
	PGPBoolean cacheGlobal;
	PGPBoolean exportable;
	PGPTime creationDate;
	PGPUInt32 expiration;
	PGPByte trustDepth;
	PGPByte trustValue;
	PGPByte *sRegExp;
	PGPSize sRegExpLength;
	PGPUInt32 *newobjs;
	PGPSize newobjsLength;

	userid = unpack_uint32(pkt);
	certifying_keyid = unpack_uint32(pkt);
	unpack_opaque(pkt, &passphrase, &passphraseLength);
	hashedPhrase = unpack_bool(pkt);
	cacheTimeOut = unpack_uint32(pkt);
	cacheGlobal = unpack_bool(pkt);
	exportable = unpack_bool(pkt);
	creationDate = unpack_uint32(pkt);
	expiration = unpack_uint32(pkt);
	trustDepth = unpack_byte(pkt);
	trustValue = unpack_byte(pkt);
	unpack_opaque(pkt, &sRegExp, &sRegExpLength);

	err = pgpCertifyUserID_back(gCtx,
			userid, certifying_keyid, (char *) passphrase, passphraseLength,
			hashedPhrase, cacheTimeOut, cacheGlobal, exportable,
			creationDate, expiration, trustDepth, trustValue,
			(char *) sRegExp, sRegExpLength,
			&newobjs, &newobjsLength);
	rpkt->status = err;
	if (IsPGPError(err)) return;

	pack_opaque(rpkt, newobjs, newobjsLength);
	if( IsntNull( newobjs ) )
		PGPFreeData( newobjs );
}

static void
sRecv_CertifyPrimaryUserID(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 userid;
	PGPByte *passphrase;
	PGPSize passphraseLength;
	PGPBoolean hashedPhrase;
	PGPUInt32 cacheTimeOut;
	PGPBoolean cacheGlobal;
	PGPUInt32 *newobjs;
	PGPSize newobjsLength;

	userid = unpack_uint32(pkt);
	unpack_opaque(pkt, &passphrase, &passphraseLength);
	hashedPhrase = unpack_bool(pkt);
	cacheTimeOut = unpack_uint32(pkt);
	cacheGlobal = unpack_bool(pkt);

	err = pgpCertifyPrimaryUserID_back(gCtx,
				userid, (char *) passphrase, passphraseLength,
				hashedPhrase, cacheTimeOut, cacheGlobal, &newobjs, &newobjsLength);
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_opaque(rpkt, newobjs, newobjsLength);
	if( IsntNull( newobjs ) )
		PGPFreeData( newobjs );
}

static void
sRecv_RevokeSig(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 sigid;
	PGPByte *passphrase;
	PGPSize passphraseLength;
	PGPBoolean hashedPhrase;
	PGPUInt32 cacheTimeOut;
	PGPBoolean cacheGlobal;
	PGPUInt32 *newobjs;
	PGPSize newobjsLength;

	sigid = unpack_uint32(pkt);
	unpack_opaque(pkt, &passphrase, &passphraseLength);
	hashedPhrase = unpack_bool(pkt);
	cacheTimeOut = unpack_uint32(pkt);
	cacheGlobal = unpack_bool(pkt);

	err = pgpRevokeSig_back(gCtx,
				sigid, (char *) passphrase, passphraseLength,
				hashedPhrase, cacheTimeOut, cacheGlobal, &newobjs, &newobjsLength);
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_opaque(rpkt, newobjs, newobjsLength);
	if( IsntNull( newobjs ) )
		PGPFreeData( newobjs );
}

static void
sRecv_RevokeKey(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 key_id;
	PGPByte *passphrase;
	PGPSize passphraseLength;
	PGPBoolean hashedPhrase;
	PGPUInt32 cacheTimeOut;
	PGPBoolean cacheGlobal;
	PGPUInt32 *newobjs;
	PGPSize newobjsLength;

	key_id = unpack_int32(pkt);
	unpack_opaque(pkt, &passphrase, &passphraseLength);
	hashedPhrase = unpack_bool(pkt);
	cacheTimeOut = unpack_uint32(pkt);
	cacheGlobal = unpack_bool(pkt);

	err = pgpRevokeKey_back(gCtx,
		key_id, (char *) passphrase, passphraseLength, hashedPhrase, cacheTimeOut, cacheGlobal,
		&newobjs, &newobjsLength);
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_opaque(rpkt, (PGPByte *)newobjs, newobjsLength);
	if( IsntNull( newobjs ) )
		PGPFreeData( newobjs );
}

static void
sRecv_DoChangePassphrase(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 keydbid, key_id, masterkey_id;
	PGPByte *oldphrase, *newphrase;
	PGPSize oldphraseLength, newphraseLength;
	PGPBoolean newpassphraseisKey;
	PGPUInt32 cacheTimeOut;
	PGPBoolean cacheGlobal;

	keydbid = unpack_uint32(pkt);
	key_id = unpack_uint32(pkt);
	masterkey_id = unpack_uint32(pkt);
	unpack_opaque(pkt, &oldphrase, &oldphraseLength);
	unpack_opaque(pkt, &newphrase, &newphraseLength);
	newpassphraseisKey = unpack_bool(pkt);
	cacheTimeOut = unpack_uint32(pkt);
	cacheGlobal = unpack_bool(pkt);

	err = pgpDoChangePassphrase_back(gCtx,
				keydbid, key_id, masterkey_id,
				(char *) oldphrase, oldphraseLength,
				(char *) newphrase, newphraseLength,
				newpassphraseisKey, cacheTimeOut, cacheGlobal);

	rpkt->status = err;
}

static void
sRecv_PassphraseIsValid(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 key_id;
	PGPByte *passphrase;
	PGPSize passphraseLength;
	PGPBoolean hashedPhrase;
	PGPBoolean isValid;
	PGPUInt32 cacheTimeOut;
	PGPBoolean cacheGlobal;

	key_id = unpack_uint32(pkt);
	unpack_opaque(pkt, &passphrase, &passphraseLength);
	hashedPhrase = unpack_bool(pkt);
	cacheTimeOut = unpack_uint32(pkt);
	cacheGlobal = unpack_bool(pkt);

	err = pgpPassphraseIsValid_back(gCtx,
				key_id, (char *) passphrase, passphraseLength,
				hashedPhrase, cacheTimeOut, cacheGlobal, &isValid);
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_bool(rpkt, isValid);
}

static void
sRecv_SetKeyTrust(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 key_id, trust;

	key_id = unpack_uint32(pkt);
	trust = unpack_uint32(pkt);

	err = pgpSetKeyTrust_back(gCtx, key_id, trust);
	rpkt->status = err;
	if (IsPGPError(err)) return;
}

static void
sRecv_GetPasskeyBuffer(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 key_id;
	PGPByte *passphrase;
	PGPSize passphraseLength;
	PGPByte *passkeyBuffer;
	PGPSize passkeyBufferLength;

	key_id = unpack_uint32(pkt);
	unpack_opaque(pkt, &passphrase, &passphraseLength);

	err = pgpGetPasskeyBuffer_back(gCtx,
				key_id, (char *) passphrase, passphraseLength,
				&passkeyBuffer, &passkeyBufferLength);
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_opaque(rpkt, passkeyBuffer, passkeyBufferLength);
	if( IsntNull( passkeyBuffer ) )
		PGPFreeData( passkeyBuffer );
}

static void
sRecv_AddKeyOptions(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 key_id;
	PGPByte *passphrase;
	PGPSize passphraseLength;
	PGPBoolean hashedPhrase;
	PGPUInt32 cacheTimeOut;
	PGPBoolean cacheGlobal;
	PGPUInt32 *raklist;
	PGPSize raklistsize;
	PGPUInt32 rakclass;
	PGPUInt32 *newobjs;
	PGPSize newobjsLength;

	key_id = unpack_uint32(pkt);
	unpack_opaque(pkt, &passphrase, &passphraseLength);
	hashedPhrase = unpack_bool(pkt);
	cacheTimeOut = unpack_uint32(pkt);
	cacheGlobal = unpack_bool(pkt);
	unpack_opaque_alloc(pkt, &raklist, &raklistsize);
	rakclass = unpack_uint32(pkt);

	err = pgpAddKeyOptions_back(gCtx,
				key_id, (char *) passphrase, passphraseLength, hashedPhrase,
				cacheTimeOut, cacheGlobal, raklist, raklistsize, rakclass, 
				&newobjs, &newobjsLength);
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_opaque(rpkt, newobjs, newobjsLength);
	if( IsntNull( newobjs ) )
		PGPFreeData( newobjs );
}

static void
sRecv_UpdateKeyOptions(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 key_id;
	PGPByte *passphrase;
	PGPSize passphraseLength;
	PGPBoolean hashedPhrase;
	PGPUInt32 cacheTimeOut;
	PGPBoolean cacheGlobal;
	PGPCipherAlgorithm *prefalg;
	PGPSize prefalgLength;
	PGPByte *prefkeyserv;
	PGPSize prefkeyservLength;
	PGPUInt32 keyflags;
	PGPBoolean fkeyflags;
	PGPUInt32 keyservprefs;
	PGPBoolean fkeyservprefs;
	PGPUInt32 *newobjs;
	PGPSize newobjsLength;

	key_id = unpack_uint32(pkt);
	unpack_opaque(pkt, &passphrase, &passphraseLength);
	hashedPhrase = unpack_bool(pkt);
	cacheTimeOut = unpack_uint32(pkt);
	cacheGlobal = unpack_bool(pkt);
	unpack_opaque(pkt, &prefalg, &prefalgLength);
	unpack_opaque(pkt, &prefkeyserv, &prefkeyservLength);
	keyflags = unpack_uint32(pkt);
	fkeyflags = unpack_bool(pkt);
	keyservprefs = unpack_uint32(pkt);
	fkeyservprefs = unpack_bool(pkt);

	err = pgpUpdateKeyOptions_back(gCtx,
				key_id, (char *) passphrase, passphraseLength,
				hashedPhrase, cacheTimeOut, cacheGlobal, prefalg, prefalgLength,
				prefkeyserv, prefkeyservLength, keyflags, fkeyflags,
				keyservprefs, fkeyservprefs, &newobjs, &newobjsLength);
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_opaque(rpkt, newobjs, newobjsLength);
	if( IsntNull( newobjs ) )
		PGPFreeData( newobjs );
}

static void
sRecv_KeyDBAddObject(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 kdb_id;
	PGPUInt32 obj_id;
	PGPUInt32 *newobjs;
	PGPSize newobjsLength;
	PGPUInt32 pnewobj;

	kdb_id = unpack_uint32(pkt);
	obj_id = unpack_uint32(pkt);

	err = pgpKeyDBAddObject_back(gCtx,
			kdb_id, obj_id, &newobjs, &newobjsLength, &pnewobj);
	rpkt->status = err;
	if (IsPGPError(err)) return;
	pack_opaque(rpkt, newobjs, newobjsLength);
	if( IsntNull( newobjs ) )
		PGPFreeData( newobjs );
	pack_uint32(rpkt, pnewobj);
}

static void
sRecv_KeyDBRemoveObject(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 kdb_id, obj_id;

	kdb_id = unpack_uint32(pkt);
	obj_id = unpack_uint32(pkt);

	err = pgpKeyDBRemoveObject_back(gCtx, kdb_id, obj_id);
	rpkt->status = err;
	if (IsPGPError(err)) return;
}

static void
sRecv_CopyKeys(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 srcid, dstid, *keylist;
	PGPSize keylistsize;
	PGPBoolean neednewkeylist;
	PGPUInt32 *newkeylist = NULL;
	PGPSize newkeylistsize;

	srcid = unpack_uint32(pkt);
	dstid = unpack_uint32(pkt);
	unpack_opaque_alloc(pkt, &keylist, &keylistsize);
	neednewkeylist = unpack_bool(pkt);

	err = pgpCopyKeys_back(gCtx,
			srcid, dstid, keylist, keylistsize, neednewkeylist,
			(neednewkeylist ? &newkeylist : NULL),
			(neednewkeylist ? &newkeylistsize : NULL) );
	rpkt->status = err;
	if (IsPGPError(err)) return;
	if( neednewkeylist )
		pack_opaque(rpkt, (PGPByte *)newkeylist, newkeylistsize);
	if( IsntNull( newkeylist ) )
		PGPFreeData( newkeylist );
}

static void
sRecv_ImportKeyBinary(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPByte *buffer;
	PGPSize length;
	PGPUInt32 kdbid, numKeys, *keyArray;
	PGPSize	keyArraySize;
	
	unpack_opaque(pkt, &buffer, &length);

	err = pgpImportKeyBinary_back(gCtx,
			buffer, length,  &kdbid, &numKeys, &keyArray, &keyArraySize);
	rpkt->status = err;
	if (IsPGPError(err)) return;

	pack_uint32(rpkt, kdbid);
	pack_uint32(rpkt, numKeys);
	pack_opaque(rpkt, keyArray, keyArraySize);
	if( IsntNull( keyArray ) )
		PGPFreeData( keyArray );
}

static void
sRecv_NewKeyDB(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 keydbid;

	(void) pkt;
	
	err = pgpNewKeyDB_back(gCtx, &keydbid);
	rpkt->status = err;
	if (IsPGPError(err)) return;

	pack_uint32(rpkt, keydbid);

	rpkt->status = err;
}

static void
sRecv_KeyDBArray(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 kdbid;
	PGPUInt32 numKeys;
	PGPUInt32 *keyArray;
	PGPSize keyArraySize;

	kdbid = unpack_uint32(pkt);

	err = pgpKeyDBArray_back(gCtx, kdbid, &numKeys, &keyArray, &keyArraySize);
	rpkt->status = err;
	if (IsPGPError(err)) return;

	pack_uint32(rpkt, numKeys);
	pack_opaque(rpkt, keyArray, keyArraySize);
	if( IsntNull( keyArray ) )
		PGPFreeData( keyArray );
}

static void
sRecv_UpdateKeyDB(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 kdbid;
	PGPUInt32 numKeys;
	PGPUInt32 *keyArray;
	PGPSize keyArraySize;
	PGPUInt32 *changedkeylist;
	PGPSize changedkeylistsize;

	kdbid = unpack_uint32(pkt);

	err = pgpUpdateKeyDB_back(gCtx, kdbid, &numKeys, &keyArray, &keyArraySize,
							  &changedkeylist, &changedkeylistsize);
	rpkt->status = err;
	if (IsPGPError(err)) return;

	pack_uint32(rpkt, numKeys);
	pack_opaque(rpkt, keyArray, keyArraySize);
	if( IsntNull( keyArray ) )
		PGPFreeData( keyArray );
	pack_opaque(rpkt, changedkeylist, changedkeylistsize);
	if( IsntNull( changedkeylist ) )
		PGPFreeData( changedkeylist );
}

static void
sRecv_SecProperties(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 id;
	PGPBoolean needsPassphrase;
	PGPBoolean isSecretShared;
	PGPCipherAlgorithm lockAlg;
	PGPUInt32 lockBits;

	id = unpack_uint32(pkt);

	err = pgpSecProperties_back(gCtx, id,
		&needsPassphrase, &isSecretShared, &lockAlg, &lockBits);
	rpkt->status = err;
	if (IsPGPError(err)) return;

	pack_bool(rpkt, needsPassphrase);
	pack_bool(rpkt, isSecretShared);
	pack_uint32(rpkt, lockAlg);
	pack_uint32(rpkt, lockBits);
}

static void
sRecv_GlobalRandomPoolAddState(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPBoolean addKeyState;
	PGPInt32 keyEvent;
	PGPBoolean addMouseState, addSystemState;

	addKeyState = unpack_bool(pkt);
	keyEvent = unpack_int32(pkt);
	addMouseState = unpack_bool(pkt);
	addSystemState = unpack_bool(pkt);

	err = pgpGlobalRandomPoolAddState_back(addKeyState,
		keyEvent, addMouseState, addSystemState );
	rpkt->status = err;
	if (IsPGPError(err)) return;
}

static void
sRecv_GlobalRandomPoolGetInfo(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 entropy, size, minEntropy;
	PGPBoolean hasMinEntropy, hasIntelRNG;

	(void) pkt;
	err = pgpGlobalRandomPoolGetInfo_back(&entropy, &size,
		&minEntropy, &hasMinEntropy, &hasIntelRNG );
	rpkt->status = err;
	if (IsPGPError(err)) return;

	pack_uint32(rpkt, entropy);
	pack_uint32(rpkt, size);
	pack_uint32(rpkt, minEntropy);
	pack_bool(rpkt, hasMinEntropy);
	pack_bool(rpkt, hasIntelRNG);
}

static void
sRecv_RandomGetBytesEntropy(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPUInt32 requestBytes, entropyBits;
	PGPByte *randBuf;
	PGPSize randBufLen;

	requestBytes = unpack_uint32(pkt);
	entropyBits = unpack_uint32(pkt);

	err = pgpRandomGetBytesEntropy_back(gCtx,
		requestBytes, entropyBits,
		&randBuf, &randBufLen);
	rpkt->status = err;
	if (IsPGPError(err)) return;

	pack_opaque(rpkt, randBuf, randBufLen);
	if( IsntNull( randBuf ) )
		PGPFreeData(randBuf);
}

static void
sRecv_RandomAddBytes(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PGPByte *buf;
	PGPSize buflen;

	unpack_opaque(pkt, &buf, &buflen);

	err = pgpRandomAddBytes_back(gCtx, buf, buflen);
	rpkt->status = err;
}

static void
sRecv_RandomStir(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;

	err = pgpRandomStir_back(gCtx);
	rpkt->status = err;
}

static void
sRecv_SetRandSeedFile(PGPPackMsg *pkt, PGPPackMsg *rpkt)
{
	PGPError err;
	PFLFileSpecRef randFile;
	
	randFile = unpack_filespec(pkt, PGPPeekContextMemoryMgr(gCtx) );

	err = pgpSetRandSeedFile_back( randFile );
	rpkt->status = err;
}

static PGPUInt32 pgpRPCAllocConnection()
{
	PGPMemoryMgrRef memref;
	struct connection *cp;

	memref = PGPGetDefaultMemoryMgr();
	if (memref == kInvalidPGPMemoryMgrRef)
		return kPGPError_UnknownError;

	cp = PGPNewData(memref, sizeof(pgpRPCconnection), kPGPMemoryMgrFlags_Clear);
	return (PGPUInt32)cp;
}


/* Handle user-initiated server shutdown */
void pgpRPCServerShuttingDown()  {
	DbgPrintf("cleaning up SDK...\n");

	PGPsdkCleanup();

	DbgPrintf("SDK cleanup done\n");
}

void
rcv_generic(PGPByte buf[], PGPSize inlen, PGPByte *outbuf[], PGPSize *outlen)
{
	PGPPackMsg pkt, rmsg;
	(void)inlen;
	
	pkt.ptr = 0;
	pkt.length = 0;
	pkt.base = buf;

	pkt.type = unpack_int32(&pkt);
	pkt.connect_ref = unpack_int32(&pkt);
	pkt.length = unpack_int32(&pkt);

	PGPGetRPCInitBuf(&rmsg.base, &rmsg.length);
	rmsg.ptr = 8;
	rmsg.status = 0;

#if PGP_UNIX
	if ((pkt.type != kPGP_Connect_MSG)
	  && !pgpRPCVerifyUnixClientAuth((PGPConnectRef)pkt.connect_ref)) {
		*(PGPInt32 *)(&rmsg.base[0]) = rmsg.status;
		*(PGPInt32 *)(&rmsg.base[4]) = rmsg.ptr;
		*outlen = rmsg.ptr;
		*outbuf = rmsg.base;
	}
#endif

	/* Save connection ref into backend context */
	if( IsntNull( gCtx ) )
		pgpContextSetConnectRef( gCtx, (PGPConnectRef) pkt.connect_ref );

#if PGP_DEBUG && PGP_WIN32
	{
		extern char *pgpRPC_cmds[];
		DbgPrintf("%-35s %08x %5d",
			pgpRPC_cmds[pkt.type],
			pkt.connect_ref,
			pkt.length );
	}
#endif
#if PGP_DEBUG && PGP_UNIX
	{
		extern char *pgpRPC_cmds[];
		extern int debug;
		if (debug) {
			printf("%-35s %08x %5d",
				pgpRPC_cmds[pkt.type],
				pkt.connect_ref,
				pkt.length );
			fflush(stdout);
		}
	}
#endif

	switch(pkt.type) {
	case kPGP_Connect_MSG:
		sRecv_Connect(&pkt, &rmsg);
		break;
	case kPGP_Disconnect_MSG:
		sRecv_Disconnect(&pkt, &rmsg);
		break;
	case kPGP_Reconnect_MSG:
		sRecv_Reconnect(&pkt, &rmsg);
		break;
	case kPGP_FetchObjectData_MSG:
		sRecv_FetchObjectData(&pkt, &rmsg);
		break;
	case kPGP_GetKeyByKeyID_MSG:
		sRecv_GetKeyByKeyID(&pkt, &rmsg);
		break;
	case kPGP_KeyEncrypt_MSG:
		sRecv_KeyEncrypt(&pkt, &rmsg);
		break;
	case kPGP_KeyDecrypt_MSG:
		sRecv_KeyDecrypt(&pkt, &rmsg);
		break;
	case kPGP_KeyVerify_MSG:
		sRecv_KeyVerify(&pkt, &rmsg);
		break;
	case kPGP_KeySign_MSG:
		sRecv_KeySign(&pkt, &rmsg);
		break;
	case kPGP_SecPassphraseOK_MSG:
		sRecv_SecPassphraseOK(&pkt, &rmsg);
		break;
	case kPGP_PurgePassphraseCache_MSG:
		sRecv_PurgePassphraseCache(&pkt, &rmsg);
		break;
	case kPGP_KeyMaxSizes_MSG:
		sRecv_KeyMaxSizes(&pkt, &rmsg);
		break;
	case kPGP_FetchKeyInfo_MSG:
		sRecv_FetchKeyInfo(&pkt, &rmsg);
		break;
	case kPGP_OpenKeyDBFile_MSG:
		sRecv_OpenKeyDBFile(&pkt, &rmsg);
		break;
	case kPGP_KeyDBFlush_MSG:
		sRecv_KeyDBFlush(&pkt, &rmsg);
		break;
	case kPGP_FreeKeyDB_MSG:
		sRecv_FreeKeyDB(&pkt, &rmsg);
		break;
	case kPGP_SetKeyEnabled_MSG:
		sRecv_SetKeyEnabled(&pkt, &rmsg);
		break;
	case kPGP_SetKeyAxiomatic_MSG:
		sRecv_SetKeyAxiomatic(&pkt, &rmsg);
		break;
	case kPGP_PropagateTrust_MSG:
		sRecv_PropagateTrust(&pkt, &rmsg);
		break;
	case kPGP_CheckKeyRingSigs_MSG:
		sRecv_CheckKeyRingSigs(&pkt, &rmsg);
		break;
	case kPGP_PrepareToCheckKeyRingSigs_MSG:
		sRecv_PrepareToCheckKeyRingSigs(&pkt, &rmsg);
		break;
	case kPGP_CheckSig_MSG:
		sRecv_CheckSig(&pkt, &rmsg);
		break;
	case kPGP_DoGenerateKey_MSG:
		sRecv_DoGenerateKey(&pkt, &rmsg);
		break;
	case kPGP_CreateKeypair_MSG:
		sRecv_CreateKeypair(&pkt, &rmsg);
		break;
	case kPGP_CreateSubkeypair_MSG:
		sRecv_CreateSubkeypair(&pkt, &rmsg);
		break;
	case kPGP_CountTokens_MSG:
		sRecv_CountTokens(&pkt, &rmsg);
		break;
    case kPGP_GetTokenInfo_MSG:
		sRecv_GetTokenInfo(&pkt, &rmsg);
		break;
	case kPGP_DeleteKeyOnToken_MSG:
		sRecv_DeleteKeyOnToken(&pkt, &rmsg);
		break;
	case kPGP_WipeToken_MSG:
		sRecv_WipeToken(&pkt, &rmsg);
		break;
	case kPGP_TokenPassphraseIsValid_MSG:
		sRecv_TokenPassphraseIsValid(&pkt, &rmsg);
		break;
	case kPGP_CopyKeyToToken_MSG:
		sRecv_CopyKeyToToken(&pkt, &rmsg);
		break;
    case kPGP_TokenImportX509_MSG:
        sRecv_TokenImportX509(&pkt, &rmsg);
        break;
    case kPGP_TokenPutKeyContainer_MSG:
        sRecv_TokenPutKeyContainer(&pkt, &rmsg);
        break;
    case kPGP_TokenGetKeyContainer_MSG:
        sRecv_TokenGetKeyContainer(&pkt, &rmsg);
        break;
    case kPGP_SetPKCS11DrvFile_MSG:
		sRecv_SetPKCS11DrvFile(&pkt, &rmsg);
		break;
	case kPGP_AddUserID_MSG:
		sRecv_AddUserID(&pkt, &rmsg);
		break;
	case kPGP_CertifyUserID_MSG:
		sRecv_CertifyUserID(&pkt, &rmsg);
		break;
	case kPGP_CertifyPrimaryUserID_MSG:
		sRecv_CertifyPrimaryUserID(&pkt, &rmsg);
		break;
	case kPGP_RevokeSig_MSG:
		sRecv_RevokeSig(&pkt, &rmsg);
		break;
	case kPGP_RevokeKey_MSG:
		sRecv_RevokeKey(&pkt, &rmsg);
		break;
	case kPGP_DoChangePassphrase_MSG:
		sRecv_DoChangePassphrase(&pkt, &rmsg);
		break;
	case kPGP_PassphraseIsValid_MSG:
		sRecv_PassphraseIsValid(&pkt, &rmsg);
		break;
	case kPGP_SetKeyTrust_MSG:
		sRecv_SetKeyTrust(&pkt, &rmsg);
		break;
	case kPGP_GetPasskeyBuffer_MSG:
		sRecv_GetPasskeyBuffer(&pkt, &rmsg);
		break;
	case kPGP_AddKeyOptions_MSG:
		sRecv_AddKeyOptions(&pkt, &rmsg);
		break;
	case kPGP_UpdateKeyOptions_MSG:
		sRecv_UpdateKeyOptions(&pkt, &rmsg);
		break;
	case kPGP_KeyDBAddObject_MSG:
		sRecv_KeyDBAddObject(&pkt, &rmsg);
		break;
	case kPGP_KeyDBRemoveObject_MSG:
		sRecv_KeyDBRemoveObject(&pkt, &rmsg);
		break;
	case kPGP_CopyKeys_MSG:
		sRecv_CopyKeys(&pkt, &rmsg);
		break;
	case kPGP_ImportKeyBinary_MSG:
		sRecv_ImportKeyBinary(&pkt, &rmsg);
		break;
	case kPGP_NewKeyDB_MSG:
		sRecv_NewKeyDB(&pkt, &rmsg);
		break;
	case kPGP_KeyDBArray_MSG:
		sRecv_KeyDBArray(&pkt, &rmsg);
		break;
	case kPGP_UpdateKeyDB_MSG:
		sRecv_UpdateKeyDB(&pkt, &rmsg);
		break;
	case kPGP_SecProperties_MSG:
		sRecv_SecProperties(&pkt, &rmsg);
		break;
	case kPGP_GlobalRandomPoolAddState_MSG:
		sRecv_GlobalRandomPoolAddState(&pkt, &rmsg);
		break;
	case kPGP_GlobalRandomPoolGetInfo_MSG:
		sRecv_GlobalRandomPoolGetInfo(&pkt, &rmsg);
		break;
	case kPGP_RandomGetBytesEntropy_MSG:
		sRecv_RandomGetBytesEntropy(&pkt, &rmsg);
		break;
	case kPGP_RandomAddBytes_MSG:
		sRecv_RandomAddBytes(&pkt, &rmsg);
		break;
	case kPGP_RandomStir_MSG:
		sRecv_RandomStir(&pkt, &rmsg);
		break;
	case kPGP_SetRandSeedFile_MSG:
		sRecv_SetRandSeedFile(&pkt, &rmsg);
		break;
	default:
		rmsg.status = kPGPError_RPCGarbledMsg;
	}

#if PGP_DEBUG && PGP_UNIX
	{
		extern int debug;
		if (debug) printf("  status=%d\n", rmsg.status);
	}
#endif

	*(PGPInt32 *)(&rmsg.base[0]) = rmsg.status;
	*(PGPInt32 *)(&rmsg.base[4]) = rmsg.ptr;
	*outlen = rmsg.ptr;
	*outbuf = rmsg.base;
}

#if PGP_DEBUG
char *pgpRPC_cmds[] = {
	"kPGP_Connect_MSG",
	"kPGP_FetchObjectData_MSG",
	"kPGP_GetKeyByKeyID_MSG",
	"kPGP_KeyEncrypt_MSG",
	"kPGP_KeyDecrypt_MSG",
	"kPGP_KeyVerify_MSG",
	"kPGP_KeySign_MSG",
	"kPGP_SecPassphraseOK_MSG",
	"kPGP_KeyMaxSizes_MSG",
	"kPGP_SecProperties_MSG",
	"kPGP_FetchKeyInfo_MSG",
	"kPGP_OpenKeyDBFile_MSG",
	"kPGP_ImportKeyBinary_MSG",
	"kPGP_KeyDBFlush_MSG",
	"kPGP_FreeKeyDB_MSG",
	"kPGP_SetKeyEnabled_MSG",
	"kPGP_SetKeyAxiomatic_MSG",
	"kPGP_PropagateTrust_MSG",
	"kPGP_CheckKeyRingSigs_MSG",
	"kPGP_PrepareToCheckKeyRingSigs_MSG",
	"kPGP_CheckSig_MSG",
	"kPGP_DoGenerateKey_MSG",
	"kPGP_AddUserID_MSG",
	"kPGP_CertifyUserID_MSG",
	"kPGP_CertifyPrimaryUserID_MSG",
	"kPGP_RevokeSig_MSG",
	"kPGP_RevokeKey_MSG",
	"kPGP_DoChangePassphrase_MSG",
	"kPGP_PassphraseIsValid_MSG",
	"kPGP_SetKeyTrust_MSG",
	"kPGP_GetPasskeyBuffer_MSG",
	"kPGP_AddKeyOptions_MSG",
	"kPGP_UpdateKeyOptions_MSG",
	"kPGP_KeyDBAddObject_MSG",
	"kPGP_KeyDBRemoveObject_MSG",
	"kPGP_NewKeyDB_MSG",
	"kPGP_KeyDBArray_MSG",
	"kPGP_GlobalRandomPoolAddState_MSG",
	"kPGP_GlobalRandomPoolGetInfo_MSG",
	"kPGP_RandomGetBytesEntropy_MSG",
	"kPGP_UpdateKeyDB_MSG",
	"kPGP_RandomAddBytes_MSG",
	"kPGP_PurgePassphraseCache_MSG",
	"kPGP_CopyKeys_MSG",
	"kPGP_SetRandSeedFile_MSG",
	"kPGP_Disconnect_MSG",
	"kPGP_Reconnect_MSG",
	"kPGP_RandomStir_MSG",
	"kPGP_CreateKeypair_MSG",
	"kPGP_CreateSubkeypair_MSG",
	"kPGP_CountTokens_MSG",
	"kPGP_GetTokenInfo_MSG",
	"kPGP_CopyKeyToToken_MSG",
	"kPGP_DeleteKeyOnToken_MSG",
	"kPGP_WipeToken_MSG",
	"kPGP_TokenPassphraseIsValid_MSG",
    "kPGP_TokenImportX509_MSG", 
    "kPGP_TokenPutKeyContainer_MSG", 
    "kPGP_TokenGetKeyContainer_MSG", 
    "kPGP_SetPKCS11DrvFile_MSG", 
};
#endif

/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
